#ifndef CONFIGURATION_H
#define CONFIGURATION_H

#include <iostream>
#include <json/json.hpp>
#include <stdexcept>

using json = nlohmann::json;

enum CoveredSetExtraction {F1, BACC, AUC};

// Static configuration object used to modifie the algorithm behaviour
// By design, all running instances of the algorithm within the same process must share the same configuration
class Configuration {
public:
    static void configure(std::istream & configuration);
    static void configure(json source);
    static std::string to_string(unsigned int spacing = 0);

    static float uncertainty_tolerance; // The maximum allowed global optimality before the optimization can terminate
    static float regularization; // The penalty incurred for each leaf inthe model
    static float upperbound; // Upperbound on the root problem for pruning problems using a greedy model

    static unsigned int time_limit; // The maximum allowed runtime (seconds). 0 means unlimited.
    static unsigned int worker_limit; // The maximum allowed worker threads. 0 means match number of available cores
    static unsigned int stack_limit; // The maximum amount of stack space (bytes) allowed to use as buffers
    static unsigned int precision_limit; // The maximum number of significant figures considered for each ordinal feature
    static unsigned int model_limit; // The maximum number of models extracted

    static bool verbose; // Flag for printing status to standard output
    static bool diagnostics; // Flag for printing diagnosis to standard output if a bug is detected

    static unsigned char depth_budget; // The maximum tree depth for solutions, counting a tree with just the root node as depth 1. 0 means unlimited.

    static unsigned int minimum_captured_points; // The mimimum captured points for any leaf

    static std::vector<int> memory_checkpoints; // Memory at which to dump trie generated by current adjency graph

    static bool objective_model_set;
    static std::vector<CoveredSetExtraction> covered_sets; 
    static std::vector<double> covered_sets_thresholds;
    static std::string covered_set_type_to_string(CoveredSetExtraction type) {
        switch (type) {
            case F1:
                return "f1";
            case BACC:
                return "bacc";
            case AUC:
                return "auc";
            default:
                throw std::invalid_argument("Unknown type");
        }
    };
    static double computeScore(CoveredSetExtraction type, unsigned int P, unsigned int N, double TP, double TN) {
        switch (type) {
            case F1:
                return TP / (TP + 0.5 * (P - TP + N - TN));
            case BACC:
                return (TP / P + TN / N) / 2;
            case AUC:
                return (TN * TP + 0.5 * (TP * (N - TN) + TN * (P - TP))) / (N * P);
            default:
                throw std::invalid_argument("Unknown type");
        }
    };

    static bool balance; // Flag for adjusting the importance of each row to equalize the total importance of each class (overrides weight)
    static bool look_ahead; // Flag for enabling the one-step look-ahead bound implemented via scopes
    static bool similar_support; // Flag for enabling the similar support bound imeplemented via the distance index
    static bool cancellation; // Flag for enabling upward propagation of cancelled subproblems
    static bool continuous_feature_exchange; // Flag for enabling the pruning of neighbouring thresholds using subset comparison
    static bool feature_exchange; // Flag for enabling the pruning of pairs of features using subset comparison
    static bool feature_transform; // Flag for enabling the equivalence discovery through simple feature transformations
    static bool rule_list; // Flag for enabling rule-list constraints on models
    static bool non_binary; // Flag for enabling non-binary encoding

    static std::string costs; // Path to file containing cost matrix
    static std::string model; // Path to file used to store the extracted models
    static std::string rashomon_model; // Path to directory used to store the Rashomon set
    static std::string rashomon_model_set; // Path to directory used to store the Rashomon model set
    static std::string rashomon_trie; // Path to directory used to store the Rashomon trie
    static std::string timing; // Path to file used to store training time
    static std::string trace; // Path to directory used to store traces
    static std::string tree; // Path to directory used to store tree-traces
    static std::string profile; // Path to file used to log runtime statistics
    static std::string datatset_encoding; // Path to file used store dataset encoding

    static bool rashomon; // Flag for extracting Rashomon set
    static float rashomon_bound_multiplier; // 
    static bool rashomon_ignore_trivial_extensions;


};

#endif